#include <unistd.h>     // For pipe, fork, read, write, close, STDOUT_FILENO, etc.
#include <stdio.h>      // For standard I/O functions like printf, fgets, snprintf
#include <stdlib.h>     // For standard library functions like exit
#include <string.h>     // For string handling functions like strlen
#include <sys/types.h>  // For data types like pid_t
#include <sys/wait.h>   // For waitpid
#include <sys/stat.h>   // For file status, permissions, etc.
#include <fcntl.h>      // For file control options like open, O_RDONLY
#include <errno.h>      // For error handling

#define MAX_LINE 80     // Maximum length of the buffer for reading and writing

// Function prototypes for client and server
void client(int writefd);
void server(int readfd);

int main(int argc, char *argv[]) {
    int pipe1[2];              // Array to store the file descriptors for the pipe
    pid_t childpid;

    pipe(pipe1);               // Create a single pipe for communication

    if ((childpid = fork()) == 0) {  // Fork a new process; if child, enter this block
        close(pipe1[1]);       // Close the write end of the pipe (used by parent)
        server(pipe1[0]);      // Call server function with the read end of the pipe
        exit(0);               // Exit child process after server is done
    }

    // Parent process block
    close(pipe1[0]);           // Close the read end of the pipe (used by child)

    client(pipe1[1]);          // Call client function with the write end of the pipe

    waitpid(childpid, NULL, 0); // Wait for the child process to terminate
    exit(0);
}

// Client function, which prompts the user for a filename and sends it to the server
void client(int writefd) {
    size_t len;
    char buff[MAX_LINE];

    printf("Enter filename: ");
    fgets(buff, MAX_LINE, stdin);  // Read user input for the filename
    len = strlen(buff);            // Get the length of the input
    if (buff[len - 1] == '\n')     // If there's a newline at the end of input
        len--;                     // Adjust length to remove the newline
    write(writefd, buff, len);     // Write the filename to the write end of the pipe
}

// Server function, which reads a filename from the client, opens the file, and sends its content to stdout
void server(int readfd) {
    int fd;
    size_t n;
    char buff[MAX_LINE + 1];

    // Read the filename/path from the client through the read end of the pipe
    if ((n = read(readfd, buff, MAX_LINE)) == 0) {
        write(STDOUT_FILENO, "EOF while reading path...\n", 26); // Error if filename is missing
        exit(0);
    }
    buff[n] = '\0';             // Null-terminate the filename string

    // Attempt to open the file in read-only mode
    if ((fd = open(buff, O_RDONLY)) < 0) {
        // If the file can't be opened, prepare an error message
        snprintf(buff + n, sizeof(buff) - n, ": can't open, %s\n", strerror(errno));
        n = strlen(buff);
        write(STDOUT_FILENO, buff, n);  // Send error message to standard output
    } else {
        // Read from the file and write its content to standard output
        while ((n = read(fd, buff, MAX_LINE)) > 0)
            write(STDOUT_FILENO, buff, n); // Output file content to standard output
        close(fd);                // Close the file after reading
    }
}

/*
1. Client reads filename from user
2. Client sends filename to pipe1
3. Server reads filename from pipe1
4. Server outputs the file content

pipe1 = client -> server (client writes, server reads)
*/ 
