#include <unistd.h>     // For pipe, fork, read, write, close, STDOUT_FILENO, etc.
#include <stdio.h>      // For standard I/O functions like printf, fgets, snprintf
#include <stdlib.h>     // For standard library functions like exit
#include <string.h>     // For string handling functions like strlen
#include <sys/types.h>  // For data types like pid_t
#include <sys/wait.h>   // For waitpid
#include <sys/stat.h>   // For file status, permissions, etc.
#include <fcntl.h>      // For file control options like open, O_RDONLY
#include <errno.h>      // For error handling

#define MAX_LINE 80     // Maximum length of the buffer for reading and writing

// Function prototypes for client and server
void client(int readfd, int writefd);
void server(int readfd, int writefd);

int main(int argc, char *argv[]) {
    int pipe1[2], pipe2[2];     // Arrays to store file descriptors for two pipes
    pid_t childpid;

    pipe(pipe1);                // Create the first pipe
    pipe(pipe2);                // Create the second pipe

    if ((childpid = fork()) == 0) {  // Fork a new process; if child, enter this block
        close(pipe1[1]);        // Close the write end of the first pipe (used by parent)
        close(pipe2[0]);        // Close the read end of the second pipe (used by parent)
        server(pipe1[0], pipe2[1]);  // Call server function with the appropriate pipe ends
        exit(0);                // Exit child process after server is done
    }

    // Parent process block
    close(pipe1[0]);            // Close the read end of the first pipe (used by child)
    close(pipe2[1]);            // Close the write end of the second pipe (used by child)

    client(pipe2[0], pipe1[1]); // Call client function with the appropriate pipe ends

    waitpid(childpid, NULL, 0); // Wait for the child process to terminate
    exit(0);
}

// Client function, which sends user input to the server and displays the server response
void client(int readfd, int writefd) {
    size_t len;
    size_t n;
    char buff[MAX_LINE];

    printf("Enter filename: ");
    fgets(buff, MAX_LINE, stdin);  // Read user input (a filename) from standard input
    len = strlen(buff);            // Get the length of the input
    if (buff[len - 1] == '\n')     // If there's a newline at the end of input
        len--;                     // Adjust length to remove the newline
    write(writefd, buff, len);     // Write the filename to the write-end of the pipe

    // Read the response from the server and output to standard output
    while ((n = read(readfd, buff, MAX_LINE)) > 0)
        write(STDOUT_FILENO, buff, n);  // Write server's response to standard output
}

// Server function, which reads a filename from the client, opens the file, and sends its content back
void server(int readfd, int writefd) {
    int fd;
    size_t n;
    char buff[MAX_LINE + 1];

    // Read the filename/path from the client through the read-end of the pipe
    if ((n = read(readfd, buff, MAX_LINE)) == 0) {
        write(writefd, "EOF while reading path...\n", 26); // Error if filename is missing
        exit(0);
    }
    buff[n] = '\0';             // Null-terminate the filename string

    // Attempt to open the file in read-only mode
    if ((fd = open(buff, O_RDONLY)) < 0) {
        // If the file can't be opened, prepare an error message
        snprintf(buff + n, sizeof(buff) - n, ": can't open, %s\n", strerror(errno));
        n = strlen(buff);
        write(writefd, buff, n);  // Send error message to the client
    } else {
        // Read from the file and write its content back to the client
        while ((n = read(fd, buff, MAX_LINE)) > 0)
            write(writefd, buff, n);  // Send file content to the client
        close(fd);                // Close the file after reading
    }
}

/*
1. Client reads filename from user
2. Client sends filename to pipe1
3. Server reads filename from pipe1
4. Server sends content to pipe2
5. Client reads content from pipe2
6. Client outputs the file content

pipe1 = client -> server
pipe2 = server -> client
*/ 