#include <unistd.h>     // For read, write, close, STDOUT_FILENO, etc.
#include <stdio.h>      // For standard I/O functions like printf, snprintf
#include <stdlib.h>     // For standard library functions like exit
#include <string.h>     // For string handling functions like strlen
#include <sys/types.h>  // For data types like size_t
#include <sys/wait.h>   // For waitpid, if used in extensions
#include <sys/stat.h>   // For file status, permissions, etc.
#include <fcntl.h>      // For file control options like open, O_RDONLY
#include <errno.h>      // For error handling

#define MAX_LINE 80               // Maximum length of the buffer for reading and writing
#define FIFO1 "/tmp/fifo.1"       // Define the path for the first named pipe (FIFO)
#define FIFO2 "/tmp/fifo.2"       // Define the path for the second named pipe (FIFO)
#define FIFO_MODE (S_IRWXU | S_IRWXG | S_IRWXO)  // Define the permissions for the FIFO

// Function prototype for the server
void server(int readfd, int writefd);

int main(int argc, char *argv[]) {
    int readfd, writefd;
    
    // Create the first FIFO with specified permissions; if it already exists, ignore the error
    if ((mkfifo(FIFO1, FIFO_MODE) < 0) && (errno != EEXIST)) {
        printf("Cannot create %s\n", FIFO1);
        exit(-1);
    }

    // Create the second FIFO; if it already exists, ignore the error
    if ((mkfifo(FIFO2, FIFO_MODE) < 0) && (errno != EEXIST)) {
        printf("Cannot create %s\n", FIFO2);
        exit(-1);
    }

    // Open FIFO1 for reading and FIFO2 for writing
    readfd = open(FIFO1, O_RDONLY);
    writefd = open(FIFO2, O_WRONLY);

    /*
        FIFO1 = client->server
        FIFO2 = server->client
    */
    

    // Call the server function with the read and write file descriptors
    server(readfd, writefd);

    exit(0);
}

// Server function, which reads a filename from the client and sends the file's content back
void server(int readfd, int writefd) {

    int fd;
    size_t n;
    char buff[MAX_LINE + 1];

    // Read the filename from the client via FIFO1
    if ((n = read(readfd, buff, MAX_LINE)) == 0) {
        write(writefd, "EOF while reading path...\n", 26);  // Error message if no filename is received
        exit(0);
    }
    buff[n] = '\0';  // Null-terminate the filename string
    printf("Received file name (%s)\n", buff);

    // Attempt to open the file in read-only mode
    if ((fd = open(buff, O_RDONLY)) < 0) {
        // If the file can't be opened, prepare an error message
        snprintf(buff + n, sizeof(buff) - n, ": can't open, %s\n", strerror(errno));
        n = strlen(buff);
        write(writefd, buff, n);  // Send error message to the client
    } else {
        printf("Sending contents of the file back to client...\n");
        // Read from the file and write its content back to the client
        while ((n = read(fd, buff, MAX_LINE)) > 0)
            write(writefd, buff, n);  // Send file content to the client
        close(fd);  // Close the file after reading
    }
}
