#include <stdio.h>
#include <mqueue.h>         // For POSIX message queue functions
#include <sys/stat.h>       // For file mode constants (e.g., S_IRWXU)
#include <stdlib.h>
#include <unistd.h>         // For getopt() and other POSIX functions
#include <time.h>           // For timestamp functions
#include <string.h>

#define MSGQOBJ_NAME    "/test2"   // Name of the message queue object
#define MAX_MSG_LEN     70         // Maximum message length

int main(int argc, char *argv[]) {
    mqd_t msgq_id;                 // Message queue descriptor
    unsigned int msgprio = 0;      // Message priority
    pid_t my_pid = getpid();       // Process ID of the current process
    char msgcontent[MAX_MSG_LEN];  // Buffer for the message content
    int create_queue = 0;          // Flag to indicate if queue should be created
    char ch;                       // Variable for getopt() option parsing
    time_t currtime;               // Variable to store the current time
    
    /* Parse command-line options "-q" and "-p prio" */
    while ((ch = getopt(argc, argv, "qp:")) != -1) {
        switch (ch) {
            case 'q':   // If "-q" is specified, set create_queue flag
                create_queue = 1;
                break;
            case 'p':   // If "-p" is specified, set the message priority
                msgprio = (unsigned int)strtol(optarg, (char **)NULL, 10); // Convert optarg to integer
                printf("I (%d) will use priority %d\n", my_pid, msgprio);
                break;
            default:    // If an unknown option is used, print usage and exit
                printf("Usage: %s [-q] -p msg_prio\n", argv[0]);
                exit(1);
        }
    }
    
    /* Ensure that "-p" is specified, as it sets the message priority */
    if (msgprio == 0) {
        printf("Usage: %s [-q] -p msg_prio\n", argv[0]);
        exit(1);
    }
    
    /* Open the message queue, creating it if the "-q" option was specified */
    if (create_queue) {
        // Create and open the queue with read-write access, and permissions for user/group
        msgq_id = mq_open(MSGQOBJ_NAME, O_RDWR | O_CREAT | O_EXCL, S_IRWXU | S_IRWXG, NULL);
    } else {
        // Open the queue with read-write access
        msgq_id = mq_open(MSGQOBJ_NAME, O_RDWR);
    }
    if (msgq_id == (mqd_t)-1) {  // If mq_open fails, print an error and exit
        perror("In mq_open()");
        exit(1);
    }

    /* Produce a message with the process ID and the current time */
    currtime = time(NULL);   // Get the current time
    snprintf(msgcontent, MAX_MSG_LEN, "Hello from process %u (at %s).", my_pid, ctime(&currtime)); // Format message
    
    /* Send the message to the queue with the specified priority */
    mq_send(msgq_id, msgcontent, strlen(msgcontent) + 1, msgprio); // Send message to the queue
    
    /* Close the message queue */
    mq_close(msgq_id);
        
    return 0;
}
