#include <stdio.h>          // For standard I/O functions like printf()
#include <sys/mman.h>       // For POSIX shared memory and memory-mapping functions
#include <sys/types.h>      // For POSIX data types
#include <unistd.h>         // For ftruncate() and other system calls
#include <fcntl.h>          // For file control options like O_CREAT, O_RDWR
#include <sys/stat.h>       // For file permissions like S_IRWXU
#include <stdlib.h>         // For standard library functions like exit()
#include <time.h>           // For time functions to seed the random number generator

// Define the path for the shared memory object
#define SHMOBJ_PATH         "/deneme123"

// Maximum length of the message content that can be stored
#define MAX_MSG_LENGTH      50

// Number of different message types supported (arbitrary example)
#define TYPES               8

// Define the structure for messages in the shared memory segment
struct msg_s {
    int type;                      // Integer to represent message type
    char content[MAX_MSG_LENGTH];  // Array to store message content
};

int main(int argc, char *argv[]) {

    if (shm_unlink(SHMOBJ_PATH) == 0)   // Delete the shared memory object
        fprintf(stderr, "Previous shared memory is deleted.\n");
    else 
        fprintf(stderr, "No shared memory is found.\n");
    
    int shmfd;                     // File descriptor for shared memory object
    int shared_seg_size = (1 * sizeof(struct msg_s)); // Define shared segment size to hold 1 message
    struct msg_s *shared_msg;      // Pointer to the shared memory segment

    // Create the shared memory object with read-write permissions, exclusive creation
    shmfd = shm_open(SHMOBJ_PATH, O_CREAT | O_EXCL | O_RDWR, S_IRWXU | S_IRWXG);
    if (shmfd < 0) {               // Check if creation failed
        perror("In shm_open()");   // Display error if opening fails
        exit(1);
    }
    fprintf(stderr, "Created shared memory object %s\n", SHMOBJ_PATH);
    
    // Adjust the size of the shared memory object to hold the message
    ftruncate(shmfd, shared_seg_size);

    // Map the shared memory object into the virtual address space
    shared_msg = (struct msg_s *)mmap(NULL, shared_seg_size, PROT_READ | PROT_WRITE, MAP_SHARED, shmfd, 0);
    if (shared_msg == NULL) {      // Check if mapping failed
        perror("In mmap()");       // Display error if mapping fails
        exit(1);
    }
    fprintf(stderr, "Shared memory segment allocated correctly (%d bytes).\n", shared_seg_size);
    
    // Seed the random number generator with the current time
    srandom(time(NULL));

    // Create a random message and store it in the shared memory segment
    shared_msg->type = random() % TYPES;   // Set a random message type
    snprintf(shared_msg->content, MAX_MSG_LENGTH, "My message, type %d, num %ld", shared_msg->type, random());

    printf("Message type is %d, content is: %s\n", shared_msg->type, shared_msg->content);
    
    return 0;
}
